const { app } = require('electron');
const fs = require('fs');
const path = require('path');
const archiver = require('archiver');
const AdmZip = require('adm-zip');
const logger = require('./logger');

const TEMPLATES_FILENAME = 'highlight-templates.json';

function getTemplatesPath() {
  return path.join(app.getPath('userData'), TEMPLATES_FILENAME);
}

function loadTemplatesFile() {
  const filePath = getTemplatesPath();
  if (!fs.existsSync(filePath)) {
    return { templates: [] };
  }
  try {
    const content = fs.readFileSync(filePath, 'utf8');
    const data = JSON.parse(content);
    return Array.isArray(data.templates) ? data : { templates: [] };
  } catch (error) {
    logger.error('[HighlightTemplate] Error loading templates:', error);
    return { templates: [] };
  }
}

function writeTemplatesFile(data) {
  const filePath = getTemplatesPath();
  fs.writeFileSync(filePath, JSON.stringify(data, null, 2), 'utf8');
}

/**
 * Save a highlight template.
 * @param {Object} template - { name, audioSample (base64), visualTemplate (object), type, threshold, minGap }
 * @returns {Object} { success: boolean, error?: string }
 */
function saveTemplate(template) {
  try {
    if (!template || !template.name || typeof template.name !== 'string') {
      return { success: false, error: 'Template name is required.' };
    }
    const name = template.name.trim();
    if (!name) {
      return { success: false, error: 'Template name cannot be empty.' };
    }

    const hasAudio = template.audioSample && typeof template.audioSample === 'string' && template.audioSample.length >= 100;
    const hasVisual = template.visualTemplate && template.visualTemplate.imageData;

    if (!hasAudio && !hasVisual) {
      return { success: false, error: 'At least one of audio or visual template is required.' };
    }

    const data = loadTemplatesFile();
    const existing = data.templates.findIndex((t) => t.name === name);
    const entry = {
      name,
      type: template.type || (hasAudio && hasVisual ? 'combined' : hasVisual ? 'visual' : 'audio'),
      audioSample: hasAudio ? template.audioSample : null,
      visualTemplate: hasVisual ? {
        imageData: template.visualTemplate.imageData,
        region: template.visualTemplate.region,
        method: template.visualTemplate.method || 'fast',
        timestamp: template.visualTemplate.timestamp,
        frameSkip: template.visualTemplate.frameSkip || 5,
        batchSize: template.visualTemplate.batchSize || 128,
        minGap: typeof template.visualTemplate.minGap === 'number' ? template.visualTemplate.minGap : (typeof template.minGap === 'number' ? template.minGap : 1)
      } : null,
      threshold: typeof template.threshold === 'number' ? template.threshold : 0.3,
      visualThreshold: typeof template.visualThreshold === 'number' ? template.visualThreshold : 0.7,
      minGap: typeof template.minGap === 'number' ? template.minGap : 1,
      createdAt: new Date().toISOString()
    };

    if (existing >= 0) {
      data.templates[existing] = entry;
    } else {
      data.templates.push(entry);
    }

    writeTemplatesFile(data);
    logger.log(`[HighlightTemplate] Saved template "${name}" (${entry.type})`);
    return { success: true };
  } catch (error) {
    logger.error('[HighlightTemplate] Error saving template:', error);
    return { success: false, error: error.message || 'Failed to save template.' };
  }
}

/**
 * Load all saved templates (metadata only, exclude large audioSample for list).
 * @returns {Object} { success: boolean, templates?: Array<{name, type, threshold, visualThreshold, minGap, createdAt}>, error?: string }
 */
function loadTemplates() {
  try {
    const data = loadTemplatesFile();
    const templates = data.templates.map((t) => ({
      name: t.name,
      type: t.type || 'audio',
      threshold: t.threshold,
      visualThreshold: t.visualThreshold,
      minGap: t.minGap,
      createdAt: t.createdAt
    }));
    return { success: true, templates };
  } catch (error) {
    logger.error('[HighlightTemplate] Error loading templates:', error);
    return { success: false, error: error.message || 'Failed to load templates.', templates: [] };
  }
}

/**
 * Load a single template by name (includes full audio sample and visual template).
 * @param {string} name - Template name
 * @returns {Object} { success: boolean, template?: Object, error?: string }
 */
function loadTemplateByName(name) {
  try {
    const data = loadTemplatesFile();
    const t = data.templates.find((x) => x.name === name);
    if (!t) {
      return { success: false, error: `Template "${name}" not found.` };
    }
    return {
      success: true,
      template: {
        name: t.name,
        type: t.type || 'audio',
        audioSample: t.audioSample,
        visualTemplate: t.visualTemplate,
        threshold: t.threshold,
        visualThreshold: t.visualThreshold,
        minGap: t.minGap,
        createdAt: t.createdAt
      }
    };
  } catch (error) {
    logger.error('[HighlightTemplate] Error loading template:', error);
    return { success: false, error: error.message || 'Failed to load template.' };
  }
}

/**
 * Delete a template by name.
 * @param {string} name - Template name
 * @returns {Object} { success: boolean, error?: string }
 */
function deleteTemplate(name) {
  try {
    const data = loadTemplatesFile();
    const idx = data.templates.findIndex((t) => t.name === name);
    if (idx < 0) {
      return { success: false, error: `Template "${name}" not found.` };
    }
    data.templates.splice(idx, 1);
    writeTemplatesFile(data);
    logger.log(`[HighlightTemplate] Deleted template "${name}"`);
    return { success: true };
  } catch (error) {
    logger.error('[HighlightTemplate] Error deleting template:', error);
    return { success: false, error: error.message || 'Failed to delete template.' };
  }
}

/**
 * Export a template to a ZIP file.
 * @param {string} name - Template name to export
 * @param {string} outputPath - Path where the ZIP file will be saved
 * @returns {Object} { success: boolean, error?: string }
 */
function exportTemplate(name, outputPath) {
  return new Promise((resolve) => {
    try {
      const data = loadTemplatesFile();
      const t = data.templates.find((x) => x.name === name);
      if (!t) {
        resolve({ success: false, error: `Template "${name}" not found.` });
        return;
      }

      const templateData = {
        name: t.name,
        type: t.type || 'audio',
        audioSample: t.audioSample,
        visualTemplate: t.visualTemplate,
        threshold: t.threshold,
        visualThreshold: t.visualThreshold,
        minGap: t.minGap,
        createdAt: t.createdAt,
        exportedAt: new Date().toISOString(),
        version: '1.1'
      };

      const output = fs.createWriteStream(outputPath);
      const archive = archiver('zip', { zlib: { level: 9 } });

      output.on('close', () => {
        logger.log(`[HighlightTemplate] Exported template "${name}" to ${outputPath}`);
        resolve({ success: true });
      });

      archive.on('error', (err) => {
        logger.error('[HighlightTemplate] Error creating export archive:', err);
        resolve({ success: false, error: err.message || 'Failed to create archive.' });
      });

      archive.pipe(output);
      archive.append(JSON.stringify(templateData, null, 2), { name: 'template.json' });
      archive.finalize();
    } catch (error) {
      logger.error('[HighlightTemplate] Error exporting template:', error);
      resolve({ success: false, error: error.message || 'Failed to export template.' });
    }
  });
}

/**
 * Import a template from a ZIP file.
 * @param {string} filePath - Path to the ZIP file
 * @returns {Object} { success: boolean, error?: string, template?: Object }
 */
function importTemplate(filePath) {
  try {
    if (!fs.existsSync(filePath)) {
      return { success: false, error: 'File not found.' };
    }

    const zip = new AdmZip(filePath);
    const templateJsonEntry = zip.getEntry('template.json');

    if (!templateJsonEntry) {
      return { success: false, error: 'Invalid template file: template.json not found.' };
    }

    const templateData = JSON.parse(templateJsonEntry.getData().toString('utf8'));

    if (!templateData.name) {
      return { success: false, error: 'Invalid template: missing name.' };
    }

    const hasAudio = templateData.audioSample && typeof templateData.audioSample === 'string' && templateData.audioSample.length >= 100;
    const hasVisual = templateData.visualTemplate && templateData.visualTemplate.imageData;

    if (!hasAudio && !hasVisual) {
      return { success: false, error: 'Invalid template: at least one of audio or visual template is required.' };
    }

    const data = loadTemplatesFile();
    let finalName = templateData.name;
    let isUpdate = false;

    const existingIdx = data.templates.findIndex((t) => t.name === finalName);
    if (existingIdx >= 0) {
      isUpdate = true;
    } else {
      let counter = 1;
      while (data.templates.some((t) => t.name === finalName)) {
        finalName = `${templateData.name} (${counter})`;
        counter++;
      }
    }

    const entry = {
      name: finalName,
      type: templateData.type || (hasAudio && hasVisual ? 'combined' : hasVisual ? 'visual' : 'audio'),
      audioSample: hasAudio ? templateData.audioSample : null,
      visualTemplate: hasVisual ? {
        ...templateData.visualTemplate,
        frameSkip: templateData.visualTemplate?.frameSkip || 5,
        batchSize: templateData.visualTemplate?.batchSize || 128
      } : null,
      threshold: typeof templateData.threshold === 'number' ? templateData.threshold : 0.3,
      visualThreshold: typeof templateData.visualThreshold === 'number' ? templateData.visualThreshold : 0.7,
      minGap: typeof templateData.minGap === 'number' ? templateData.minGap : 1,
      createdAt: templateData.createdAt || new Date().toISOString()
    };

    if (isUpdate) {
      data.templates[existingIdx] = entry;
    } else {
      data.templates.push(entry);
    }

    writeTemplatesFile(data);
    logger.log(`[HighlightTemplate] Imported template "${finalName}"${isUpdate ? ' (updated)' : ''} from ${filePath}`);

    return {
      success: true,
      template: entry,
      message: isUpdate ? `Template "${finalName}" updated.` : `Template "${finalName}" imported.`
    };
  } catch (error) {
    logger.error('[HighlightTemplate] Error importing template:', error);
    return { success: false, error: error.message || 'Failed to import template.' };
  }
}

module.exports = {
  saveTemplate,
  loadTemplates,
  loadTemplateByName,
  deleteTemplate,
  exportTemplate,
  importTemplate
};
